<?php

namespace Tangibledesign\Framework\Traits;

use Tangibledesign\Framework\Core\Collection;
use Tangibledesign\Framework\Models\Helpers\HasMeta;
use Tangibledesign\Framework\Models\Review;

trait HasReviewsTrait
{
    use HasMeta;

    abstract public function getId(): int;

    abstract public function getReviewType(): string;

    public function getReviewNumber(): int
    {
        $cacheKey = tdf_prefix() . '/reviews/' . $this->getReviewType() . '/number/' . $this->getId();
        $cachedNumber = get_transient($cacheKey);

        if ($cachedNumber !== false) {
            return $cachedNumber;
        }

        $number = tdf_query_reviews()
            ->model($this->getId(), $this->getReviewType())
            ->getResultsNumber();

        set_transient($cacheKey, $number, 60 * 60 * 24);

        return $number;
    }

    public function getReviewCountByRating(int $rating): int
    {
        return tdf_query_reviews()
            ->model($this->getId(), $this->getReviewType())
            ->filterByRating($rating)
            ->getResultsNumber();
    }

    public function getReviews(int $page = 1, int $limit = 10): Collection
    {
        return tdf_query_reviews()
            ->model($this->getId(), $this->getReviewType())
            ->take($limit)
            ->skip($page * $limit - $limit)
            ->orderByNewest()
            ->get();
    }

    public function getRating(): string
    {
        $cacheKey = tdf_prefix() . '/reviews/' . $this->getReviewType() . '/rating/' . $this->getId();
        $cachedRating = get_transient($cacheKey);

        if ($cachedRating !== false) {
            return $cachedRating;
        }

        $rating = $this->calculateRating();

        $formattedRating = number_format($rating, 1, tdf_settings()->getDecimalSeparator(), '');

        set_transient($cacheKey, $formattedRating, 60 * 60 * 24);

        return $formattedRating;
    }

    private function calculateRating(): float
    {
        $rating = tdf_query_reviews()
            ->model($this->getId(), $this->getReviewType())
            ->get()
            ->average(static function ($review) {
                /** @var Review $review */
                return $review->getRating();
            });

        return round($rating, 1);
    }
}